<?php

namespace App\Http\Controllers;

use App\Models\File;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class FileController extends Controller
{
    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:png,jpg,gif,mp4,mov,avi,wmv,flv,mkv|max:20480',
        ],[
            'file.required' => 'Seleciona uma capa',
            'file.mimes' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].',
            'file.max' => 'O tamanho máximo do arquivo é 20MB.'
        ]);

        $id = trim($request->input('id'));       
        $filetype = trim($request->input('file_type'));

        if (!$request->file('file')->isValid()) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo não é válido.']);
        }
    
        $file = $request->file('file');
        $filename = time() . '_' . $file->getClientOriginalName();
    
        $extension = strtolower($file->getClientOriginalExtension());
        $allowed_extensions = ['png', 'jpg', 'gif', 'mp4', 'mov', 'avi', 'wmv', 'flv', 'mkv'];

        if (!in_array($extension, $allowed_extensions)) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].']);
        }


        try {                        

            $files = new File([
                'file' => $filename,
                'file_type2' => $filetype,
                'extension_file2' => $extension,
                'project_id' => $id
            ]);

            if($files->save()){
                Storage::disk('public')->put('uploads/project_' .$id.'/'.$filename, file_get_contents($file));
            }
            self::InsertLogs("Adicionou uma imagem num projecto");
            return redirect()->back()->with(['status' => 'success', 'msg' => 'Ficheiro registado com sucesso!']);
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível registar o ficheiro, tente novamente! ".$e->getMessage()]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
        $project = Project::find($filteredId);
        $totalRegistros = File::where('project_id', $filteredId)->count();
        
        $files = File::select('files.*')
        ->join('projects', 'projects.idProject', '=', 'files.project_id')
        ->where('files.project_id', '=', $filteredId)
        ->get();
        
        if(is_null($project)){
            return redirect()->route('projectos')->with(['status' => 'error', 'msg' => 'Não existe projecto com este ID']);
        }
        $data = $this->dataUser();       
        return view('admin.projects.addImages',['data'=>$files, 'total'=>$totalRegistros, 'id'=>$filteredId, 'dataUser' => $data]);
    }

    public function destroy(int $id)
    {
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $file = File::find($filteredId);
            if($file){
                if (Storage::disk('public')->exists('uploads/project_'.$filteredId.'/' . $file->file)) {
                    Storage::disk('public')->delete('uploads/project_'.$filteredId.'/' . $file->file);
                    self::InsertLogs("Eliminou uma imagem de um projecto.");
                    $file->delete();                    
                    return redirect()->back()->with(['status' => 'success', 'msg' => 'Ficheiro eliminado com sucesso!']);
                }
            }
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível eliminar o ficheiro, tente novamente! "]);
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível eliminar o ficheiro, tente novamente! "]);
        }
    }
}
