<?php

namespace App\Http\Controllers;

use App\Models\File;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $projects = Project::orderBy('created_at','DESC')->get();
        self::InsertLogs("Visualizou todos os projectos registados.");
        return view('admin.projects.projects',['data' => $projects]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {    
        self::InsertLogs("Abriu o formulário de registar projectos.");
        return view('admin.projects.add');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $send = new NewsletterController();

        $request->validate([
            'name_project' => 'required|string|unique:projects',
            'category' => 'required|string',
            'file' => 'required|file|mimes:png,jpg,gif,mp4,mov,avi,wmv,flv,mkv|max:20480',
        ],[
            'name_project.required' => 'O campo nome é obrigatório.',
            'name_project.unique' => 'Já existe um projecto com este nome.',
            'category.required' => 'O campo categoria é obrigatório.',
            'file.required' => 'Seleciona uma capa',
            'file.mimes' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].',
            'file.max' => 'O tamanho máximo do arquivo é 10MB.'
        ]);
        $category = trim($request->input('category'));
        $name = trim($request->input('name_project'));
        $client = trim($request->input('client'));
        $desc = trim($request->input('description'));      
        $filetype = trim($request->input('file_type'));
        $category_en = trim($request->input('category_en'));
        $desc_en = trim($request->input('description_en'));  
                
        $name_projet = Project::where('name_project','=',$name)
        ->first();

        if ($name_projet) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'Já existe um projecto com este nome.']);
        }

        if (!$request->file('file')->isValid()) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo não é válido.']);
        }
    
        $file = $request->file('file');
        $filename = time() . '_' . $file->getClientOriginalName();
    
        $extension = strtolower($file->getClientOriginalExtension());
        $allowed_extensions = ['png', 'jpg', 'gif', 'mp4', 'mov', 'avi', 'wmv', 'flv', 'mkv'];

        if (!in_array($extension, $allowed_extensions)) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].']);
        }


        try {                        

            $project = new Project([
                'category' => $category,
                'category_en' => $category_en,
                'name_project' => $name,
                'client' => $client,
                'description_project' => $desc,
                'description_project_en' => $desc_en,
                'thumb' => $filename,
                'file_type' => $filetype,
                'extension_file' => $extension,
                'slug' => Str::slug($name.'-'.substr(uniqid(), 10, 12)),
                'user_id' => Auth::user()->id
            ]);

            if ($project->save()) {
                // Criar o nome do diretório com base no ID do projeto
                $projectDirectory = 'uploads/project_' . $project->idProject.'/';
        
                // Verificar se o diretório já existe, caso contrário, criar
                if (!Storage::disk(name: 'public')->exists($projectDirectory)) {
                    Storage::disk('public')->makeDirectory($projectDirectory);
                }
                        
                Storage::disk('public')->put('uploads/project_'.$project->idProject.'/'.$filename, file_get_contents($file));            
                self::InsertLogs("Registou um projecto com o nome [$name na categoria $category]");
                $send->sendNewsletterToSubscribers(null, $name, $category, $desc);
                return redirect()->back()->with(['status' => 'success', 'msg' => 'Projecto registado com sucesso!']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível registar o projecto, tente novamente! "]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
        $project = Project::find($filteredId);
        if(is_null($project)){
            return redirect()->route('projectos')->with(['status' => 'error', 'msg' => 'Não existe projecto com este ID']);
        }
        if(!$project){
            return redirect()->route('projectos')->with(['status' => 'error', 'msg' => 'Não existe projecto com este ID']);
        }
        self::InsertLogs("Visualizou os detalhes do projecto [$project->name_project com a categoria $project->category]");
        return view('admin.projects.addImages',['data'=>$project]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(int $id)
    {
        $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
        return view('admin.projects.addProject',['project'=>Project::find($filteredId)]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id)
    {

        $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
        $project = Project::find($filteredId);

        if(!$project){
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'Não foi possível encontrar este projecto.']);
        }

        $request->validate([
            'name_project' => 'required|string',
            'category' => 'required|string'
        ],[
            'name_project.required' => 'O campo nome é obrigatório.',
            'category.required' => 'O campo categoria é obrigatório.'      
        ]);

        if(!empty($request->file('file'))){
            $request->validate([
                'name_project' => 'required|string',
                'category' => 'required',
                'category´_en' => 'required',
                'file' => 'required|file|mimes:png,jpg,gif,mp4,mov,avi,wmv,flv,mkv|max:20480',
            ],[
                'name_project.required' => 'O campo nome é obrigatório.',
                'category.required' => 'O campo categoria é obrigatório.',
                'file.required' => 'Seleciona uma capa',
                'file.mimes' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].',
                'file.max' => 'O tamanho máximo do arquivo é 10MB.'
            ]);
        }

        $id = trim($request->input('id'));
        $category = trim($request->input('category'));
        $category_en = trim($request->input('category_en'));
        $name = trim($request->input('name_project'));
        $client = trim($request->input('client'));
        $desc = trim($request->input('description'));      
        $desc_en = trim($request->input('description_en'));      
        $filetype = trim($request->input('file_type'));  
        $file = $request->file('file'); 
                
        $name_projet = Project::where('name_project',$name)
        ->where('idProject','!=',$id)
        ->first();

        if ($name_projet) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'Já existe um projecto com este nome.']);
        }

        
        if(!empty($file)){
            if (!$request->file('file')->isValid()) {
                return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo não é válido.']);
            }
                    
            $filename = time() . '_' . $file->getClientOriginalName();
        
            $extension = strtolower($file->getClientOriginalExtension());
            $allowed_extensions = ['png', 'jpg', 'gif', 'mp4', 'mov', 'avi', 'wmv', 'flv', 'mkv'];
    
            if (!in_array($extension, $allowed_extensions)) {
                return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo deve ser [PNG, JPG, GIF, MP4, MOV, AVI, WMV, FLV, MKV].']);
            }
        }
        
        try {                        

            $result = $project->update([
                'category' => $category,
                'category_en' => $category_en,
                'name_project' => $name,
                'client' => $client,
                'description_project' => $desc,
                'description_project_en' => $desc_en,
                'thumb' => (empty($file)) ? $project->thumb : $filename,
                'file_type' => (empty($file)) ? $project->file_type : $filetype,
                'extension_file' => (empty($file)) ? $project->extension_file : $extension,
            ]);

            if ($result) {    
                if(!empty($file)){
                    // Criar o nome do diretório com base no ID do projeto            
                    $projectDirectory = 'public/uploads/project_' . $project->idProject.'/';
                    // Verificar se o diretório já existe, caso contrário, criar
                    if (!Storage::disk('public')->exists($projectDirectory)) {
                        Storage::disk('public')->makeDirectory($projectDirectory);
                    }
                    // Mover o arquivo para o diretório específico do projeto
                    $file->storeAs($projectDirectory, $filename);
                }
                                    
                self::InsertLogs("Editou os dados do projecto com id [$id]");
                return redirect()->back()->with(['status' => 'success', 'msg' => 'Dados actualizados com sucesso!']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível registar o projecto, tente novamente! ".$e->getMessage()]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id)
    {
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $project = Project::find($filteredId);
            if ($project) {
                
                $projectDirectory = 'uploads/project_' . $filteredId;
        
                if (Storage::disk('public')->exists($projectDirectory)) {
                    
                    Storage::disk('public')->deleteDirectory($projectDirectory);
        
                    
                    File::where('project_id', $filteredId)->delete();
                            
                    self::InsertLogs("Eliminou um projecto com o nome [$project->name_project na categoria $project->category]");
                    $project->delete();
                    return redirect()->back()->with(['status' => 'success', 'msg' => 'Projeto e arquivos relacionados deletados com sucesso!']);
                } else {
                    return redirect()->back()->with(['status' => 'error', 'msg' => 'O diretório do projeto não existe.']);
                }
            } else {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Projeto não encontrado.']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível eliminar o projecto, tente novamente! "]);
        }
    }
    
}
