<?php

namespace App\Http\Controllers;

use App\Models\Team;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TeamController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view('admin.team.team', ['data' => Team::all()]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.team.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'especiality' => 'required|string',
            'file' => 'required|file|mimes:png,jpg,gif,hic|max:20480',
        ],[
            'name.required' => 'O campo nome é obrigatório.',
            'especiality.required' => 'O campo especialidade é obrigatório.',
            'file.required' => 'Seleciona uma fotografia',
            'file.mimes' => 'O arquivo deve ser [PNG, JPG, GIF, HEIC].',
            'file.max' => 'O tamanho máximo do arquivo é 10MB.'
        ]);
        $name = trim($request->input('name'));
        $especiality = trim($request->input('especiality'));   

        if (!$request->file('file')->isValid()) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo não é válido.']);
        }
    
        $file = $request->file('file');
        $filename = time() . '_' . $file->getClientOriginalName();
    
        $extension = strtolower($file->getClientOriginalExtension());
        $allowed_extensions = ['png', 'jpg', 'gif', 'heic'];

        if (!in_array($extension, $allowed_extensions)) {
            return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo deve ser [PNG, JPG, GIF, HEIC].']);
        }


        try {                        

            $team = new Team([
                'image' => $filename,
                'name' => $name,
                'especiality' => $especiality,
            ]);

            if ($team->save()) {
                // Criar o nome do diretório com base no ID do projeto
                $directory = 'uploads/team/';
        
                // Verificar se o diretório já existe, caso contrário, criar
                if (!Storage::disk(name: 'public')->exists($directory)) {
                    Storage::disk('public')->makeDirectory($directory);
                }
                        
                Storage::disk('public')->put($directory.$filename, file_get_contents($file));            
                self::InsertLogs("Registou um membro com o nome [$name com a especialidade $especiality]");                
                return redirect()->back()->with(['status' => 'success', 'msg' => 'Membro registado com sucesso!']);
            }
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível registar o membro, tente novamente! "]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Team $team)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(int $id)
    {
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $team = Team::find($filteredId);

            if (!$team || is_null($filteredId)) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Membro não encontrado não encontrado.']);
            }            
            return view('admin.team.create',['team' => $team]);
        }catch(Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => 'Não foi possível eliminar este utilizador.']);            
        } 
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id)
    {
        $team = Team::find($id);

        if(!$team) return redirect()->back()->withErrors('Membro não encontrado.');

        $request->validate([
            'name' => 'required|string',
            'especiality' => 'required|string',
            'file' => 'required|file|mimes:png,jpg,gif,hic|max:20480',
        ],[
            'name.required' => 'O campo nome é obrigatório.',
            'especiality.required' => 'O campo especialidade é obrigatório.',
            'file.required' => 'Seleciona uma fotografia',
            'file.mimes' => 'O arquivo deve ser [PNG, JPG, GIF, HEIC].',
            'file.max' => 'O tamanho máximo do arquivo é 10MB.'
        ]);

        $name = trim($request->input('name'));
        $especiality = trim($request->input('especiality'));   

        if(!empty($request->file('file'))){
            if (!$request->file('file')->isValid()) {
                return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo não é válido.']);
            }
        
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
        
            $extension = strtolower($file->getClientOriginalExtension());
            $allowed_extensions = ['png', 'jpg', 'gif', 'heic'];
            
            if (!in_array($extension, $allowed_extensions)) {
                return redirect()->back()->withInput()->withErrors(['status' => 'error', 'msg' => 'O arquivo deve ser [PNG, JPG, GIF, HEIC].']);
            }
            $directory = 'uploads/team/';
            Storage::disk('public')->put($directory.$filename, file_get_contents($file)); 
        }else{
            $filename = $team->image;
        }
        
        try {                        

            $team->update([
                'image' => $filename,
                'name' => $name,
                'especiality' => $especiality,
            ]);
            self::InsertLogs("Actualizou os dados do membro com ID $id");                
            return redirect()->back()->with(['status' => 'success', 'msg' => 'Dados actualizados com sucesso!']);            
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível actualizar os dados, tente novamente! "]);
        }

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id)
    {        
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $team = Team::find($filteredId);

            if (!$team || is_null($filteredId)) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Membro não encontrado não encontrado.']);
            }
            $team->delete();
            return redirect()->back()->with(['status' => 'success', 'msg' => 'Membro eliminado com sucesso.']);
        }catch(Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => 'Não foi possível eliminar este membro.']);            
        }        
    }
}
