<?php

namespace App\Http\Controllers;

use App\Mail\Reset;
use App\Models\Project;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::all();
        self::InsertLogs("Visualizou todos os utilizadores do sistema");
        return view('admin.users.users',['data' => $users]);
    }

    public function create(){
        $dataUser = $this->dataUser();
        self::InsertLogs("Abriu o formulário para registar utilizador");
        return view('admin.users.add',['dataUser' => $dataUser]);        
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $input = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email',            
            'password' => 'required|min:6',
            'level' => 'required|string',
        ],[
            'name.required' => 'O campo nome é obrigatório.',
            'email.required' => 'O campo email é obrigatório.',
            'email.unique' => 'Já existe um utilizador com este email'
        ]);
        
        try {            

            $user = User::where('email', $input['email'])->first();
            if($user){
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Já existe um utilziador com este e-mail.']);
            }

            $user = new User([
                'name' => trim($input['name']),
                'email' => trim($input['email']),
                'password' => bcrypt($input['password']),
                'level' => $input['level']
            ]);

            $user->save();
            self::InsertLogs("Registou um novo utilizador [".$input['name']." - ".$input['email']."]");
            return redirect()->back()->with(['status' => 'success', 'msg' => 'Utilizador registado com sucesso!']);
        } catch (\Exception $e) {
            self::InsertLogs("Tentou registar um utlizador");
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possível registar este utilizador, tente novamente! "]);
        }   
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
        $user = User::find($filteredId);
        if(!$user || is_null($user)){
            return redirect()->route('utilizadores')->with(['status' => 'error', 'msg' => 'Não existe utilizador com este ID']);
        }
        return view('admin.users.add',['data'=>$user]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id)
    {
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $user = User::find($filteredId);

            if (!$user || is_null($filteredId)) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Este utilizador não existe']);
            }

            $user->status_user = $request->input('status')==1 ? 0 : 1;
            $user->save();
            self::InsertLogs(($request->input('status')==1) ? "Desactivou a conta do utilizador [$user->name - $user->email]":"Activou a conta do utilizador [$user->name - $user->email]");
            return redirect()->back()->with(['status' => 'success', 'msg' => ($request->input('status')==1) ? 'Conta desactivada com sucesso.' : 'Conta activada com sucesso.']);

        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possivel efectuar a operação, tente novamente."]);
        } 
    }

    public function reset(int $id)
    {
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $user = User::find($filteredId);

            if (!$user || is_null($filteredId)) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Este utilizador não existe']);
            }

            $password = self::generatePassword();

            $user->update([
                'password' => $password
            ]);

            self::InsertLogs("Resetou a palavra passe do utilizador [$user->name - $user->email]");
            Mail::to($user->email)->send(new Reset([
                'name' => $user->name,
                'password' => $password
            ]));
            return redirect()->back()->with(['status' => 'success', 'msg' => 'Palavra passe reseta com sucesso.']);

        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => "Não foi possivel efectuar a operação, tente novamente. ".$e->getMessage()]);
        } 
    }

    public function resetar(Request $request)
    {
        try {
            $email = $request->input('email');
            $user = User::where('email', $email)->first();
            $name = $email;

            if (!$user) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Este utilizador não existe']);
            }

            $password = self::generatePassword();

            $user->update([
                'password' => $password
            ]);

            self::insertLogs("[$name - $email] resetou a sua palavra passe");

            Mail::to($email)->send(new Reset([
                'name' => $name,
                'password' => $password
            ]));

            return redirect()->back()->with(['status' => 'success', 'msg' => 'Solicitação efetuada com sucesso, verifique a sua caixa de email.']);

        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => 'Não foi possível efetuar a operação, tente novamente. '.$e->getMessage()]);
        } 
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id)
    {        
        try{
            $filteredId = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $user = User::find($filteredId);

            if (!$user || is_null($filteredId)) {
                return redirect()->back()->with(['status' => 'error', 'msg' => 'Este utilizador não existe']);
            }

            $project = Project::where('user_id', $filteredId)->get();
            
            if($project->count()==0){
                self::InsertLogs("Eliminou o utilizador [$user->name - $user->email]");
                $user->delete();
                return redirect()->back()->with(['status' => 'success', 'msg' => 'Utilizador eliminado.']);
            }
            return redirect()->back()->with(['status' => 'error', 'msg' => 'Não foi possível eliminar este utilizador, existem projectos associados a ele, se pretende remover o acesso para este utilizador, desactiva a sua conta.']);
        }catch(Exception $e) {
            return redirect()->back()->with(['status' => 'error', 'msg' => 'Não foi possível eliminar este utilizador.']);      
        }        
    }

}
